/*
 * SPDX license identifier: MPL-2.0
 *
 * Copyright (C) 2023 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology and Bosch.
 *
 * This file is part of COVESA Project DLT - Diagnostic Log and Trace.
 *
 * This Source Code Form is subject to the terms of the
 * Mozilla Public License (MPL), v. 2.0.
 * If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * For further information see https://covesa.global.
 */

/*!
 * \author
 * Le Van Khanh <Khanh.LeVan@vn.bosch.com>
 *
 * \copyright Copyright © 2023 Advanced Driver Information Technology. \n
 * License MPL-2.0: Mozilla Public License version 2.0 http://mozilla.org/MPL/2.0/.
 *
 * \file dlt_daemon_statistic.c
 */

#if defined(DLT_LOG_STATISTIC) && defined(BOSCH_ECP_543_RMAP_2452492_LOG_ENABLE_SPAM_FILTERING)
#include "dlt_daemon_statistic.h"

/* LogLevel string representation */
DLT_STATIC char *log_level_str[DLT_LOG_MAX] = {
    [DLT_LOG_OFF] = (char*)"OFF",
    [DLT_LOG_FATAL] = (char*)"FATAL",
    [DLT_LOG_ERROR] = (char*)"ERROR",
    [DLT_LOG_WARN] = (char*)"WARNING",
    [DLT_LOG_INFO] = (char*)"INFO",
    [DLT_LOG_DEBUG] = (char*)"DEBUG",
    [DLT_LOG_VERBOSE] = (char*)"VERBOSE"
};

#define DLT_LOG_STATISTICS_START                    DLT_LOG_FATAL
#define DLT_LOG_STATISTICS_END                      DLT_LOG_VERBOSE

/*TABLEN for time info column width*/
#define TABLEN_1                                    15
/*TABLEN for context statistic column width*/
#define TABLEN_2                                    (2*TABLEN_1)
#define DLT_LOG_STATISTICS_MAX_PRINT_LENGTH         256
/*Convert bytes to kilobytes*/
#define DLT_STAT_BYTES_TO_KBYTES_DIVISOR            1024
#define NSEC_TO_USEC                                1000

#define FPRINTF_CHECK(pFile, ...) \
    do { \
        if (fprintf(pFile, __VA_ARGS__) < 0) { \
            dlt_vlog(LOG_ERR, "%s: Write failed (errno: %d - %s)\n", \
                     __func__, errno, strerror(errno)); \
            fclose(pFile); \
            return -1; \
        } \
    } while(0)

DLT_STATIC DltDaemon *daemon_internal;
DLT_STATIC DltDaemonLocal *daemon_local_internal;

DLT_STATIC char statistics_filename[NAME_MAX + 1] = STR_UNDEFINED;
DLT_STATIC const char *space_str = " ";
static int tablength = TABLEN_1;

typedef enum {
    DLT_STAT_CTX_PRINT_ECU = 0,
    DLT_STAT_CTX_PRINT_APP,
    DLT_STAT_CTX_PRINT_CTX,
    DLT_STAT_CTX_PRINT_MSG_TOTAL,
    DLT_STAT_CTX_PRINT_MSG_PER_SEC,
    DLT_STAT_CTX_PRINT_SIZE_PER_SEC,
    DLT_STAT_CTX_PRINT_MAX
} DltDaemonStatisticsCtxPrint;

DLT_STATIC char *ctx_print_element[DLT_STAT_CTX_PRINT_MAX] = {
    (char*)"EcuID",
    (char*)"AppID",
    (char*)"Ctx",
    (char*)"Message/total",
    (char*)"Message/sec",
    (char*)"kByte/sec"
};

typedef enum {
    DLT_STAT_TIME_PRINT_TIME = 0,
    DLT_STAT_TIME_PRINT_START_TIME,
    DLT_STAT_TIME_PRINT_END_TIME,
    DLT_STAT_TIME_PRINT_DURATION,
    DLT_STAT_TIME_PRINT_MAX
} DltDaemonStatisticsTimePrint;

DLT_STATIC char *time_print_element[DLT_STAT_TIME_PRINT_MAX] = {
    (char*)"    ",
    (char*)"Start time",
    (char*)"End time",
    (char*)"Duration (sec)"
};

/**
 * @brief Convert string to number of tabs
 *
 * @param str   string input
 *
 * @return number of tabs.
 */
DLT_STATIC int dlt_daemon_statistic_get_next_tab_from_string(char *str)
{
    if (str == NULL || tablength <= 0) {
        return 0;
    }

    return (int)(strlen(str) / (size_t)tablength) + 1;
}

/**
 * @brief Add spaces to string
 *
 * Appends the specified number of space characters to the buffer pointed to by text.
 * The buffer must be null-terminated and have enough capacity for additional spaces.
 * Precondition: space_str must be a single space character (global constant).
 *
 * @param text          Pointer to buffer string (must be null-terminated)
 * @param num           Number of spaces to add (size_t for consistency)
 * @param buffer_size   Total capacity of buffer
 * @return 0 on success, -1 on error (buffer overflow or invalid parameters)
 */
DLT_STATIC int dlt_daemon_statistic_add_spaces_to_string(char **text, int num, size_t buffer_size)
{
    char *text_ptr;
    size_t current_len;
    size_t space_len;
    int i;

    if ((text == NULL) || (*text == NULL)) {
        dlt_vlog(LOG_WARNING, "%s: Invalid input parameters\n", __func__);
        return -1;
    }
    if (buffer_size == 0 || buffer_size > DLT_LOG_STATISTICS_MAX_PRINT_LENGTH) {
        dlt_vlog(LOG_ERR, "%s: Invalid buffer size %zu\n", __func__, buffer_size);
        return -1;
    }
    /* Nothing to do */
    if (num <= 0) {
        return 0;
    }

    text_ptr = *text;
    current_len = strnlen(text_ptr, buffer_size);
    space_len = strlen(space_str);

    /* Add spaces one by one with bounds checking */
    for (i = 0; i < num; i++) {
        /* Check if we have enough space */
        if (current_len + space_len + 1 > buffer_size) {
            dlt_vlog(LOG_WARNING, "%s: Buffer overflow prevented after adding %d of %d spaces\n",
                     __func__, i, num);
            return -1;
        }

        /* Add single space */
        size_t remaining_space = buffer_size - current_len;
        snprintf(text_ptr + current_len, remaining_space, "%s", space_str);
        current_len += space_len;
    }
    return 0;
}

/**
 * @brief Print statistics information to log specific
 *
 * @param str   The string to print
 *
 * @return 0 on success, -1 on error
 */
DLT_STATIC int dlt_daemon_statistic_print_to_log_specific(char *str)
{
    if (str == NULL) {
        dlt_vlog(LOG_ERR, "%s: Input string is NULL\n", __func__);
        return -1;
    }

    if (dlt_log(LOG_INFO, str) != DLT_RETURN_OK) {
        dlt_vlog(LOG_ERR, "%s: Failed to write to standard DLT log\n", __func__);
        return -1;
    }

    if ((daemon_internal != NULL) && (daemon_local_internal != NULL)) {
        if (dlt_daemon_log_internal_statistics(daemon_internal, daemon_local_internal, str,
                                               daemon_local_internal->flags.vflag) != 0) {
            dlt_vlog(LOG_ERR, "%s: Failed to log internal statistic\n", __func__);
            return -1;
        }
    }
    else {
        dlt_vlog(LOG_ERR, "%s: Daemon internal logging not available (daemon_internal=%p, daemon_local_internal=%p)\n",
                 __func__, (void*)daemon_internal, (void*)daemon_local_internal);
        return -1;
    }
    return 0;
}

/**
 * @brief Function to append value and pad with spaces
 *
 * @param print_data        Buffer to append to
 * @param print_data_size   Size of the buffer
 * @param value             Value to append
 * @param next_tab_pos      Next tab position for padding
 *
 * @return 0 on success, -1 on error
 */
static int dlt_daemon_statistic_append_and_pad(char *print_data, size_t print_data_size, const char *value, int next_tab_pos)
{
    if (print_data == NULL) {
        dlt_vlog(LOG_ERR, "%s: print_data parameter is NULL\n", __func__);
        return -1;
    }
    if (print_data_size == 0) {
        dlt_vlog(LOG_ERR, "%s: print_data_size is zero\n", __func__);
        return -1;
    }
    if (print_data_size > DLT_LOG_STATISTICS_MAX_PRINT_LENGTH) {
        dlt_vlog(LOG_ERR, "%s: print_data_size %zu exceeds maximum %d\n",
                 __func__, print_data_size, DLT_LOG_STATISTICS_MAX_PRINT_LENGTH);
        return -1;
    }

    if (value == NULL) {
        dlt_vlog(LOG_ERR, "%s: value parameter is NULL\n", __func__);
        return -1;
    }

    if (next_tab_pos < 0) {
        dlt_vlog(LOG_ERR, "%s: Invalid next_tab_pos %d (must be >= 0)\n",
                 __func__, next_tab_pos);
        return -1;
    }

    size_t current_len = strlen(print_data);
    size_t value_len = strlen(value);
    size_t remaining = (current_len < print_data_size) ? print_data_size - current_len : 0;
    if (remaining > value_len) {
        snprintf(print_data + current_len, remaining, "%s", value);
    } else {
        dlt_vlog(LOG_ERR, "%s: Buffer overflow prevented - insufficient space for value\n", __func__);
        return -1;
    }
    char *print_ptr = print_data;
    size_t subline_len = strlen(print_data);
    int spaces_needed = (next_tab_pos * tablength) - (int)subline_len;
    if (spaces_needed < 0) {
        dlt_vlog(LOG_WARNING, "%s: Buffer overflow prevented - spaces calculation resulted in negative value\n", __func__);
        return -1;
    }
    if (dlt_daemon_statistic_add_spaces_to_string(&print_ptr, spaces_needed, print_data_size) != 0) {
        dlt_vlog(LOG_DEBUG, "%s: Failed to add spaces to string - buffer capacity exceeded\n", __func__);
        return -1;
    }
    return 0;
}

/**
 * @brief Format time header line for statistics output
 *
 * @return 0 on success, -1 on error
 */
DLT_STATIC int dlt_daemon_statistic_format_time_line(void)
{
    char log_str[DLT_DAEMON_TEXTBUFSIZE];
    char print_element[DLT_LOG_STATISTICS_MAX_PRINT_LENGTH] = {0};
    /* Format the header line with column names */
    for (int i = DLT_STAT_TIME_PRINT_TIME; i < DLT_STAT_TIME_PRINT_MAX; i++) {
        int next_tab_pos = dlt_daemon_statistic_get_next_tab_from_string(print_element);
        if (dlt_daemon_statistic_append_and_pad(print_element, sizeof(print_element), time_print_element[i], next_tab_pos) != 0) {
            return -1;
        }
    }
    snprintf(log_str, sizeof(log_str), "%s\n", print_element);
    if (dlt_daemon_statistic_print_to_log_specific(log_str) != 0) {
        dlt_vlog(LOG_ERR, "%s: Failed to print time line to log\n", __func__);
        return -1;
    }
    return 0;
}

/**
 * @brief Print time information line (generic helper)
 *
 * Formats and prints a time information line with label, start time, end time, and duration.
 * This is a generic helper used by both monotonic and system time printing functions.
 *
 * @param time_label            Label for the time type (e.g., "Monotonic time", "System time")
 * @param time_start            Time start timestamp
 * @param time_end              Time end timestamp
 * @param stat_time_duration    Time period duration in seconds
 *
 * @return 0 on success, -1 on error
 */
static int dlt_daemon_statistic_print_time_line(const char *time_label,
                                                 struct timespec time_start,
                                                 struct timespec time_end,
                                                 uint32_t stat_time_duration)
{
    char log_str[DLT_DAEMON_TEXTBUFSIZE];
    char print_element[DLT_LOG_STATISTICS_MAX_PRINT_LENGTH] = {0};
    char subline[DLT_LOG_STATISTICS_MAX_PRINT_LENGTH] = {0};
    int next_tab_pos = -1;
    char* print_ptr = NULL;
    size_t subline_len = 0;
    int spaces_needed = 0;


    /* Validate input parameter: time_label must not be NULL */
    if (time_label == NULL) {
        dlt_vlog(LOG_ERR, "%s: time_label parameter is NULL\n", __func__);
        return -1;
    }
    /* Validate time_label is not empty */
    if (time_label[0] == '\0') {
        dlt_vlog(LOG_WARNING, "%s: time_label is empty\n", __func__);
        return -1;
    }

    for (int i = DLT_STAT_TIME_PRINT_TIME; i < DLT_STAT_TIME_PRINT_MAX; i++) {
        next_tab_pos = dlt_daemon_statistic_get_next_tab_from_string(print_element);
        memset(subline, 0, sizeof(subline));

        if (i == DLT_STAT_TIME_PRINT_TIME) {
            snprintf(subline, sizeof(subline), "%s", time_label);
        } else if (i == DLT_STAT_TIME_PRINT_START_TIME) {
            snprintf(subline, sizeof(subline), "%5u.%06u",
                    (unsigned int)time_start.tv_sec,
                    (unsigned int)(time_start.tv_nsec/NSEC_TO_USEC));
        } else if (i == DLT_STAT_TIME_PRINT_END_TIME) {
            snprintf(subline, sizeof(subline), "%5u.%06u",
                    (unsigned int)time_end.tv_sec,
                    (unsigned int)(time_end.tv_nsec/NSEC_TO_USEC));
        } else if (i == DLT_STAT_TIME_PRINT_DURATION) {
            snprintf(subline, sizeof(subline), "%u", stat_time_duration);
        }

        size_t current_len = strlen(print_element);
        size_t remaining = (current_len < sizeof(print_element)) ? sizeof(print_element) - current_len : 0;
        if (remaining > strlen(subline)) {
            snprintf(print_element + current_len, remaining, "%s", subline);
        } else {
            dlt_vlog(LOG_WARNING, "%s: Buffer overflow prevented for %s element %d\n",
                     __func__, time_label, i);
            return -1;
        }
        print_ptr = print_element;
        subline_len = strlen(print_element);
        spaces_needed = (next_tab_pos * tablength) - (int)subline_len;
        if (spaces_needed < 0) {
            dlt_vlog(LOG_WARNING, "%s: Buffer overflow prevented - spaces calculation resulted in negative value\n",
                __func__);
            return -1;
        }
        if (dlt_daemon_statistic_add_spaces_to_string(&print_ptr, spaces_needed, sizeof(print_element)) != 0) {
            dlt_vlog(LOG_DEBUG, "%s: Failed to add spaces to string - buffer capacity exceeded\n", __func__);
            return -1;
        }
    }

    snprintf(log_str, sizeof(log_str), "%s\n", print_element);
    if (dlt_daemon_statistic_print_to_log_specific(log_str) != 0) {
        dlt_vlog(LOG_ERR, "%s: Failed to print %s line to log\n", __func__, time_label);
        return -1;
    }
    return 0;
}

/**
 * @brief Print monotonic time information line
 *
 * @param m_start               Monotonic time start
 * @param m_end                 Monotonic time end
 * @param stat_time_duration    Time period since the start of statistical information collection
 *
 * @return 0 on success, -1 on error
 */
DLT_STATIC int dlt_daemon_statistic_print_monotonic_time(struct timespec m_start, struct timespec m_end, uint32_t stat_time_duration)
{
    return dlt_daemon_statistic_print_time_line("Monotonic time", m_start, m_end, stat_time_duration);
}

/**
 * @brief Print system time information line
 *
 * @param s_start               System time start
 * @param s_end                 System time end
 * @param stat_time_duration    Time period since the start of statistical information collection
 *
 * @return 0 on success, -1 on error
 */
DLT_STATIC int dlt_daemon_statistic_print_system_time(struct timespec s_start, struct timespec s_end, uint32_t stat_time_duration)
{
    return dlt_daemon_statistic_print_time_line("System time", s_start, s_end, stat_time_duration);
}

/**
 * @brief Print time header information
 *
 * @param m_start               Monotonic time start
 * @param m_end                 Monotonic time end
 * @param s_start               System time start
 * @param s_end                 System time end
 * @param stat_time_duration    Time period since the start of statistical information collection
 *
 * @return 0 on success, -1 on error
 */
DLT_STATIC int dlt_daemon_statistic_print_time_info(struct timespec m_start, struct timespec m_end,
    struct timespec s_start, struct timespec s_end,
    uint32_t stat_time_duration)
{
    /* Save current table width and validate */
    int saved_tablength = tablength;

    if (tablength <= 0) {
        dlt_vlog(LOG_WARNING, "%s: Invalid tablength (%d), using default\n", __func__, tablength);
        tablength = TABLEN_1;
        saved_tablength = tablength;
        return -1;
    }

    /* Use wider columns for time data (timestamps need more space) */
    tablength = TABLEN_2;

    /* Print time information sections */
    if (dlt_daemon_statistic_format_time_line() != 0) {
        dlt_vlog(LOG_WARNING, "%s: Failed to print time header\n", __func__);
        return -1;
    }
    if (dlt_daemon_statistic_print_monotonic_time(m_start, m_end, stat_time_duration) != 0) {
        dlt_vlog(LOG_WARNING, "%s: Failed to print monotonic time\n", __func__);
        return -1;
    }
    if (dlt_daemon_statistic_print_system_time(s_start, s_end, stat_time_duration) != 0) {
        dlt_vlog(LOG_WARNING, "%s: Failed to print system time\n", __func__);
        return -1;
    }

    /* Restore original table width for context statistics (narrower columns) */
    tablength = saved_tablength;
    return 0;
}

/**
 * @brief Print context statistics header
 *
 * @return 0 on success, -1 on error
 */
DLT_STATIC int dlt_daemon_statistic_print_context_header(void)
{
    char log_str[DLT_DAEMON_TEXTBUFSIZE];
    char print_element[DLT_LOG_STATISTICS_MAX_PRINT_LENGTH] = {0};
    int ret = 0;
    int next_tab_pos = -1;

    /* Print section header */
    snprintf(log_str, sizeof(log_str), "Output (Ctx):\n");
    if (dlt_daemon_statistic_print_to_log_specific(log_str) != 0) {
        dlt_vlog(LOG_ERR, "%s: Failed to print statistics info to log specific\n", __func__);
        return -1;
    }

    /* Print ECU, AppID, Ctx headers */
    for (int i = DLT_STAT_CTX_PRINT_ECU; i <= DLT_STAT_CTX_PRINT_CTX; i++) {
        next_tab_pos = dlt_daemon_statistic_get_next_tab_from_string(print_element);
        if (next_tab_pos < 0) {
            dlt_vlog(LOG_ERR, "%s: Invalid tab position for element %d\n", __func__, i);
            return -1;
        }

        ret = dlt_daemon_statistic_append_and_pad(print_element, sizeof(print_element),
                                                   ctx_print_element[i], next_tab_pos);
        if (ret != 0) {
            dlt_vlog(LOG_ERR, "%s: Failed to append context header element %d (%s)\n",
                     __func__, i, ctx_print_element[i]);
            return -1;
        }
    }

    /* Print log level headers */
    for (int i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
        next_tab_pos = dlt_daemon_statistic_get_next_tab_from_string(print_element);
        if (next_tab_pos < 0) {
            dlt_vlog(LOG_ERR, "%s: Invalid tab position for log level %d\n", __func__, i);
            return -1;
        }
        ret = dlt_daemon_statistic_append_and_pad(print_element, sizeof(print_element),
                                                   log_level_str[i], next_tab_pos);
        if (ret != 0) {
            dlt_vlog(LOG_ERR, "%s: Failed to append log level header %s\n",
                     __func__, log_level_str[i]);
            return -1;
        }
    }

    /* Print message/total, message/sec, kByte/sec headers */
    for (int i = DLT_STAT_CTX_PRINT_MSG_TOTAL; i < DLT_STAT_CTX_PRINT_MAX; i++) {
        next_tab_pos = dlt_daemon_statistic_get_next_tab_from_string(print_element);
        if (next_tab_pos < 0) {
            dlt_vlog(LOG_ERR, "%s: Invalid tab position for aggregate element %d\n", __func__, i);
            return -1;
        }

        ret = dlt_daemon_statistic_append_and_pad(print_element, sizeof(print_element),
                                                   ctx_print_element[i], next_tab_pos);
        if (ret != 0) {
            dlt_vlog(LOG_ERR, "%s: Failed to append aggregate header element %d (%s)\n",
                     __func__, i, ctx_print_element[i]);
            return -1;
        }
    }

    snprintf(log_str, DLT_DAEMON_TEXTBUFSIZE, "%s\n", print_element);
    if (dlt_daemon_statistic_print_to_log_specific(log_str) != 0) {
        dlt_vlog(LOG_ERR, "%s: Failed to print context header to log\n", __func__);
        return -1;
    }
    return 0;
}

/**
 * @brief Print statistics for a single context
 *
 * @param user_list             Registered users list
 * @param context               Context to print statistics for
 * @param application           Application containing the context
 * @param stat_time_duration    Time period for rate calculations
 *
 * @return 0 on success, -1 on error
 */
DLT_STATIC int dlt_daemon_statistic_print_single_context(DltDaemonRegisteredUsers *user_list,
                                                          DltDaemonContext *context,
                                                          DltDaemonApplication *application,
                                                          uint32_t stat_time_duration)
{
    char log_str[DLT_DAEMON_TEXTBUFSIZE];
    char print_data[DLT_LOG_STATISTICS_MAX_PRINT_LENGTH] = { 0 };
    int i = 0;
    uint32_t stat_cnt = 0;
    uint32_t stat_size = 0;

    // Print ECU, AppID, Context ID
    for (i = DLT_STAT_CTX_PRINT_ECU; i <= DLT_STAT_CTX_PRINT_CTX; i++) {
        char value[8] = {0};
        if (i == DLT_STAT_CTX_PRINT_ECU) {
            snprintf(value, sizeof(value), "%.4s", user_list->ecu);
        } else if (i == DLT_STAT_CTX_PRINT_APP) {
            snprintf(value, sizeof(value), "%.4s", application->apid);
        } else if (i == DLT_STAT_CTX_PRINT_CTX) {
            snprintf(value, sizeof(value), "%.4s", context->ctid);
        }
        int next_tab_pos = dlt_daemon_statistic_get_next_tab_from_string(print_data);
        if (dlt_daemon_statistic_append_and_pad(print_data, sizeof(print_data), value, next_tab_pos) != 0) {
            return -1;
        }
    }

    // Print log level statistics
    for (i = (int)DLT_LOG_STATISTICS_START; i <= (int)DLT_LOG_STATISTICS_END; i++) {
        char value[16] = {0};
        snprintf(value, sizeof(value), "%u", context->log_level_statistics[i].stat_cnt);
        int next_tab_pos = dlt_daemon_statistic_get_next_tab_from_string(print_data);
        if (dlt_daemon_statistic_append_and_pad(print_data, sizeof(print_data), value, next_tab_pos) != 0) {
            return -1;
        }
        stat_cnt += context->log_level_statistics[i].stat_cnt;
        stat_size += context->log_level_statistics[i].stat_size;
    }

    // Calculate rates
    uint32_t stat_cnt_per_sec = 0;
    uint32_t stat_size_per_sec = 0;
    if (stat_time_duration != 0) {
        stat_cnt_per_sec = stat_cnt / stat_time_duration;
        stat_size_per_sec = (stat_size / DLT_STAT_BYTES_TO_KBYTES_DIVISOR) / stat_time_duration;
    }

    // Print totals and rates
    for (i = DLT_STAT_CTX_PRINT_MSG_TOTAL; i < DLT_STAT_CTX_PRINT_MAX; i++) {
        char value[16] = {0};
        if (i == DLT_STAT_CTX_PRINT_MSG_TOTAL) {
            snprintf(value, sizeof(value), "%u", stat_cnt);
        } else if (i == DLT_STAT_CTX_PRINT_MSG_PER_SEC) {
            snprintf(value, sizeof(value), "%u", stat_cnt_per_sec);
        } else if (i == DLT_STAT_CTX_PRINT_SIZE_PER_SEC) {
            snprintf(value, sizeof(value), "%u", stat_size_per_sec);
        }
        int next_tab_pos = dlt_daemon_statistic_get_next_tab_from_string(print_data);
        if (dlt_daemon_statistic_append_and_pad(print_data, sizeof(print_data), value, next_tab_pos) != 0) {
            return -1;
        }
    }

    snprintf(log_str, sizeof(log_str), "%s\n", print_data);
    if (dlt_daemon_statistic_print_to_log_specific(log_str) != 0) {
        dlt_vlog(LOG_ERR, "%s: Failed to print statistics info to log specific\n", __func__);
    }
    return 0;
}

/**
 * @brief Print statistics for all contexts
 *
 * @param user_list             Registered users list
 * @param stat_time_duration    Time period for rate calculations
 *
 * @return 0 on success, -1 on error
 */
DLT_STATIC int dlt_daemon_statistic_print_all_contexts(DltDaemonRegisteredUsers *user_list, uint32_t stat_time_duration)
{
    DltDaemonApplication *application = NULL;
    DltDaemonContext *context = NULL;
    int offset_base = 0;
    int app_idx = 0;
    int prev_app_idx = 0;
    int ctx_idx = 0;
    int context_index = 0;
    int num_contexts = 0;

    if ((user_list == NULL) || (user_list->num_applications <= 0) ||
       (user_list->applications == NULL) || (user_list->contexts == NULL)) {
        dlt_vlog(LOG_ERR, "%s: user_list is invalid (NULL pointer or invalid count).\n", __func__);
        return -1;
    }

    for (app_idx = 0; app_idx < user_list->num_applications; app_idx++) {
        application = &user_list->applications[app_idx];
        offset_base = 0;

        /* Calculate start offset within contexts[] for this application */
        for (prev_app_idx = 0; prev_app_idx < app_idx; prev_app_idx++) {
            num_contexts = user_list->applications[prev_app_idx].num_contexts;
            if (num_contexts > INT_MAX - offset_base) {
                dlt_vlog(LOG_ERR, "%s: Integer overflow at app %d (offset: %d, adding: %d)\n",
                         __func__, app_idx, offset_base, num_contexts);
                return -1;
            }
            offset_base += num_contexts;
        }
        if (offset_base < 0 || offset_base > user_list->num_contexts) {
            dlt_vlog(LOG_ERR, "%s: Invalid offset %d for app %d (max: %d)\n",
                     __func__, offset_base, app_idx, user_list->num_contexts);
            continue;
        }

        /* Iterate through all contexts in this application */
        for (ctx_idx = 0; ctx_idx < application->num_contexts; ctx_idx++) {
            context_index = offset_base + ctx_idx;
            if (context_index < 0 || context_index >= user_list->num_contexts) {
                dlt_vlog(LOG_ERR, "%s: Context index %d out of bounds (max: %d)\n",
                         __func__, context_index, user_list->num_contexts - 1);
                break;
            }
            context = &(user_list->contexts[context_index]);
            if (context) {
                dlt_daemon_statistic_print_single_context(user_list, context, application, stat_time_duration);
            }
        }
    }
    return 0;
}

/**
 * @brief Print statistics information to log (refactored version)
 *
 * @param user_list             Registered users list
 * @param m_start               Monotonic time start
 * @param m_end                 Monotonic time end
 * @param s_start               System time start
 * @param s_end                 System time end
 * @param stat_time_duration    Time period since the start of statistical information collection
 *
 *  @return 0 on success, -1 on error
 */
int dlt_daemon_statistic_print_to_log(DltDaemonRegisteredUsers *user_list,
                                       struct timespec m_start, struct timespec m_end,
                                       struct timespec s_start, struct timespec s_end,
                                       uint32_t stat_time_duration)
{
    tablength = TABLEN_1;

    if ((user_list == NULL) || (user_list->num_applications <= 0)||
        (user_list->applications == NULL) || (user_list->contexts == NULL)) {
        dlt_vlog(LOG_ERR, "%s: user_list is invalid (NULL pointer or invalid count).\n", __func__);
        return -1;
    }

    /* Print time information header: monotonic time, system time, and durations */
    if (dlt_daemon_statistic_print_time_info(m_start, m_end, s_start, s_end, stat_time_duration) != 0) {
        dlt_vlog(LOG_WARNING, "%s: Failed to print time info\n", __func__);
        return -1;
    }

    /* Print context statistics header: ECU/App/Context IDs and column labels */
    if (dlt_daemon_statistic_print_context_header() != 0) {
        dlt_vlog(LOG_WARNING, "%s: Failed to print context header\n", __func__);
        return -1;
    }

    /* Print statistics for all registered contexts: iterate through apps and their contexts */
    if (dlt_daemon_statistic_print_all_contexts(user_list, stat_time_duration) != 0) {
        dlt_vlog(LOG_WARNING, "%s: Failed to print statistic data for all context\n", __func__);
        return -1;
    }

    /* Print final separator newline (optional operation: fail gracefully) */
    if (dlt_daemon_statistic_print_to_log_specific((char*)"\n") != 0) {
        dlt_vlog(LOG_WARNING, "%s: Failed to print separator newline to log\n", __func__);
    }
    return 0;
}


/**
 * @brief Helper to write CSV header and time information
 *
 * @param pFile                 File pointer to write to
 * @param m_start               Monotonic time start
 * @param m_end                 Monotonic time end
 * @param s_start               System time start
 * @param s_end                 System time end
 * @param stat_time_duration    Time period duration
 *
 * @return 0 on success, -1 on error
 */
static int dlt_daemon_statistic_write_csv_header_and_time(FILE *pFile,
                                                           struct timespec m_start,
                                                           struct timespec m_end,
                                                           struct timespec s_start,
                                                           struct timespec s_end,
                                                           uint32_t stat_time_duration)
{
    FPRINTF_CHECK(pFile, ",Start time,End time,Duration (sec),,,,\n");
    FPRINTF_CHECK(pFile, "Monotonic time,%5u.%06u,%5u.%06u,%u,,\n",
                (unsigned int)m_start.tv_sec,
                (unsigned int)(m_start.tv_nsec/NSEC_TO_USEC),
                (unsigned int)m_end.tv_sec,
                (unsigned int)(m_end.tv_nsec/NSEC_TO_USEC),
                stat_time_duration);
    FPRINTF_CHECK(pFile, "System time,%5u.%06u,%5u.%06u,%u,,\n",
                (unsigned int)s_start.tv_sec,
                (unsigned int)(s_start.tv_nsec/NSEC_TO_USEC),
                (unsigned int)s_end.tv_sec,
                (unsigned int)(s_end.tv_nsec/NSEC_TO_USEC),
                stat_time_duration);
    return 0;
}

/**
 * @brief Helper to write context statistics header
 *
 * @param pFile     File pointer to write to
 *
 * @return 0 on success, -1 on error
 */
static int dlt_daemon_statistic_write_context_header(FILE *pFile) {
    int log_idx;
    FPRINTF_CHECK(pFile, "Output (Ctx):,,\n");
    FPRINTF_CHECK(pFile, "EcuID,AppID,Ctx,");
    for (log_idx = (int)DLT_LOG_STATISTICS_START; log_idx <= (int)DLT_LOG_STATISTICS_END; log_idx++) {
        FPRINTF_CHECK(pFile, "%s,", log_level_str[log_idx]);
    }
    FPRINTF_CHECK(pFile, "Message/total,Message/sec,kByte/sec,,\n");
    return 0;
}

/**
 * @brief Helper to write a single context's statistics
 *
 * @param pFile                 File pointer to write to
 * @param user_list             Registered users list
 * @param application           Application containing the context
 * @param context               Context to write statistics for
 * @param stat_time_duration    Time period for rate calculations
 *
 * @return 0 on success, -1 on error
 */
static int dlt_daemon_statistic_write_context_stats(FILE *pFile,
                                                     DltDaemonRegisteredUsers *user_list,
                                                     DltDaemonApplication *application,
                                                     DltDaemonContext *context,
                                                     uint32_t stat_time_duration)
{
    int log_idx;
    uint32_t stat_cnt = 0;
    uint32_t stat_size = 0;
    uint32_t stat_cnt_per_sec = 0;
    uint32_t stat_size_per_sec = 0;
    if (!context) {
        return 0;
    }
    FPRINTF_CHECK(pFile, "%.4s,%.4s,%.4s,", user_list->ecu, application->apid, context->ctid);
    for (log_idx = (int)DLT_LOG_STATISTICS_START; log_idx <= (int)DLT_LOG_STATISTICS_END; log_idx++) {
        FPRINTF_CHECK(pFile, "%u,", context->log_level_statistics[log_idx].stat_cnt);
        stat_cnt += context->log_level_statistics[log_idx].stat_cnt;
        stat_size += context->log_level_statistics[log_idx].stat_size;
    }
    if (stat_time_duration != 0) {
        stat_cnt_per_sec = stat_cnt / stat_time_duration;
        stat_size_per_sec = (stat_size / DLT_STAT_BYTES_TO_KBYTES_DIVISOR) / stat_time_duration;
    }
    FPRINTF_CHECK(pFile, "%u,%u,%u,\n", stat_cnt, stat_cnt_per_sec, stat_size_per_sec);
    return 0;
}

/**
 * @brief Print statistics information to CSV file
 *
 * @param user_list             Registered users list (contains applications and contexts)
 * @param m_start               Monotonic time start timestamp
 * @param m_end                 Monotonic time end timestamp
 * @param s_start               System time start timestamp
 * @param s_end                 System time end timestamp
 * @param stat_time_duration    Time period duration in seconds for rate calculations
 *
 * @return 0 on success, -1 on error (file I/O failure, invalid parameters, overflow)
 */
DLT_STATIC int dlt_daemon_statistic_print_to_file(DltDaemonRegisteredUsers *user_list,
                                        struct timespec m_start, struct timespec m_end,
                                        struct timespec s_start, struct timespec s_end,
                                        uint32_t stat_time_duration)
{
    FILE *pFile = NULL;
    int app_idx = 0;
    int prev_app_idx = 0;
    int offset_base = 0;
    int ctx_idx = 0;
    int context_index = 0;
    int num_contexts = 0;
    DltDaemonApplication *application = NULL;
    DltDaemonContext *context = NULL;

    if ((statistics_filename[0] == '\0') || (strcmp(statistics_filename, STR_UNDEFINED) == 0)) {
        dlt_log(LOG_ERR, (char*)"Statistics filename not set or is undefined\n");
        return -1;
    }
    if ((user_list == NULL) || (user_list->num_applications <= 0) ||
        (user_list->applications == NULL) || (user_list->contexts == NULL)) {
        dlt_vlog(LOG_ERR, "%s: user_list is invalid (NULL pointer or invalid count).\n", __func__);
        return -1;
    }
    pFile = fopen(statistics_filename, "w+");
    if (pFile == NULL) {
        dlt_vlog(LOG_ERR, "%s: Cannot open file for writing: %s (errno: %d - %s)\n",
                 __func__, statistics_filename, errno, strerror(errno));
        return -1;
    }
    if (dlt_daemon_statistic_write_csv_header_and_time(pFile, m_start, m_end, s_start, s_end, stat_time_duration) != 0) {
        fclose(pFile);
        return -1;
    }
    if (dlt_daemon_statistic_write_context_header(pFile) != 0) {
        fclose(pFile);
        return -1;
    }
    if (user_list->num_applications > 0) {
        for (app_idx = 0; app_idx < user_list->num_applications; app_idx++) {
            application = &user_list->applications[app_idx];
            offset_base = 0;
            for (prev_app_idx = 0; prev_app_idx < app_idx; prev_app_idx++) {
                num_contexts = user_list->applications[prev_app_idx].num_contexts;
                if (num_contexts > INT_MAX - offset_base) {
                    dlt_vlog(LOG_ERR, "%s: Integer overflow at app %d (offset: %d, adding: %d)\n",
                            __func__, app_idx, offset_base, num_contexts);
                    fclose(pFile);
                    return -1;
                }
                offset_base += num_contexts;
            }
            if ((offset_base < 0) || (offset_base > user_list->num_contexts)) {
                dlt_vlog(LOG_ERR, "%s: Invalid offset %d for app %d (max: %d)\n",
                        __func__, offset_base, app_idx, user_list->num_contexts);
                continue;
            }
            for (ctx_idx = 0; ctx_idx < application->num_contexts; ctx_idx++) {
                context_index = offset_base + ctx_idx;
                if ((context_index < 0) || (context_index >= user_list->num_contexts)) {
                    dlt_vlog(LOG_ERR, "%s: Context index %d out of bounds (max: %d)\n",
                            __func__, context_index, user_list->num_contexts - 1);
                    break;
                }
                context = &(user_list->contexts[context_index]);
                if (dlt_daemon_statistic_write_context_stats(pFile, user_list, application, context, stat_time_duration) != 0) {
                    fclose(pFile);
                    return -1;
                }
            }
        }
    }
    FPRINTF_CHECK(pFile, "\n");
    if (fclose(pFile) != 0) {
        dlt_vlog(LOG_ERR, "%s: Failed to close file (errno: %d - %s)\n",
                 __func__, errno, strerror(errno));
        return -1;
    }
    return 0;
}


/**
 * @brief Set internal statistics dlt-daemon reference
 *
 * @param daemon pointer to dlt daemon structure
 * @param daemon_local pointer to dlt daemon local structure
 */
void dlt_daemon_statistic_set_daemon_internal(DltDaemon *daemon, DltDaemonLocal *daemon_local)
{
    daemon_internal = daemon;
    daemon_local_internal = daemon_local;
}

/**
 * @brief Set the output filename for statistics CSV export
 *
 * This function configures the CSV filename where statistics will be written.
 * It validates the filename length and safely copies it to the internal buffer.
 *
 * @param filename  Path to the CSV output file (must not be NULL)
 * @return 0 on success, -1 on error
 */
int dlt_daemon_statistic_set_filename(const char *filename)
{
    size_t filename_len = 0;

    /* Validate input: filename pointer must not be NULL */
    if (filename == NULL) {
        dlt_log(LOG_ERR, (char*)"Wrong parameter: filename is NULL\n");
        return -1;
    }

    if (strnlen(filename, NAME_MAX) == 0) {
        dlt_log(LOG_ERR, (char*)"Wrong parameter: filename is empty\n");
        return -1;
    }

    /* Check filename length to prevent buffer overflow */
    filename_len = strlen(filename);
    if (filename_len >= sizeof(statistics_filename)) {
        dlt_vlog(LOG_ERR, "Filename too long (%zu bytes), maximum allowed is %zu bytes\n",
                filename_len, sizeof(statistics_filename) - 1);
        return -1;
    }

    /* Safe copy with explicit null termination */
    snprintf(statistics_filename, sizeof(statistics_filename), "%s", filename);
    return 0;
}

/**
 * @brief Print statistics information to log and CSV file
 *
 * This is the main entry point for exporting DLT statistics. It calculates the
 * duration between start and end timestamps, then exports statistics to both
 * the DLT log and CSV file (if configured).
 * @param user_list             Registered users list
 * @param m_start               Monotonic time start
 * @param m_end                 Monotonic time end
 * @param s_start               System time start
 * @param s_end                 System time end
 * @return 0 on success, -1 on error
 */
int dlt_daemon_statistic_print(DltDaemonRegisteredUsers *user_list,
                                struct timespec m_start, struct timespec m_end,
                                struct timespec s_start, struct timespec s_end)
{
    uint32_t stat_time_duration = 0;

    if ((user_list == NULL) || (user_list->num_applications <= 0) ||
        (user_list->applications == NULL) || (user_list->contexts == NULL)) {
        dlt_vlog(LOG_ERR, "%s:  user_list is invalid (NULL pointer or invalid count).\n", __func__);
        return -1;
    }

    /* Validate that end time is not before start time */
    if ((m_end.tv_sec < m_start.tv_sec) ||
        (m_end.tv_sec == m_start.tv_sec && m_end.tv_nsec < m_start.tv_nsec)) {
        dlt_vlog(LOG_ERR, "%s: invalid time range (end < start).\n", __func__);
        return -1;
    }

    /* Calculate duration in seconds (rounded down) */
    if (m_end.tv_nsec >= m_start.tv_nsec) {
        stat_time_duration = m_end.tv_sec - m_start.tv_sec;
    } else {
        /* Borrow 1 second for nanosecond underflow */
        stat_time_duration = (m_end.tv_sec - 1) - m_start.tv_sec;
    }

    /* Reject zero-duration periods (minimum 1 second required for meaningful rate calculations) */
    if (stat_time_duration == 0) {
        dlt_vlog(LOG_WARNING, "%s: Duration validation failed: duration is 0 seconds (minimum 1 second required).\n", __func__);
        return -1;
    }
    if (dlt_daemon_statistic_print_to_log(user_list, m_start, m_end, s_start, s_end, stat_time_duration) != 0) {
        dlt_vlog(LOG_ERR, "%s: Failed to print statistic to log.\n", __func__);
        return -1;
    }
    if (dlt_daemon_statistic_print_to_file(user_list, m_start, m_end, s_start, s_end, stat_time_duration) != 0) {
        dlt_vlog(LOG_ERR, "%s: Failed to print statistic to file.\n", __func__);
        return -1;
    }

    return 0;
}

/**
 * @brief Reset statistics information
 *
 * This function resets all statistical counters (message count and size) for every
 * context across all applications. It iterates through all registered applications
 * and their contexts, setting both stat_cnt and stat_size to zero for each log level.
 *
 * @param user_list  Pointer to registered users list containing applications and contexts
 * @return 0 on success, -1 on error
 */
int dlt_daemon_statistic_reset(DltDaemonRegisteredUsers *user_list)
{
    DltDaemonApplication *application = NULL;
    DltDaemonContext *context = NULL;
    int app_idx = 0;
    int prev_app_idx = 0;
    int offset_base = 0;
    int ctx_idx = 0;
    int context_index = 0;
    int num_contexts = 0;
    int log_idx = 0;

    /* Validate input: user_list structure and essential components must be non-NULL and valid */
    if ((user_list == NULL) || (user_list->num_applications <= 0) ||
        (user_list->applications == NULL) || (user_list->contexts == NULL)) {
        dlt_vlog(LOG_ERR, "%s:  user_list is invalid (NULL pointer or invalid count).\n", __func__);
        return -1;
    }

    /* Iterate through all applications and reset their contexts */
    if (user_list->num_applications > 0) {
        for (app_idx = 0; app_idx < user_list->num_applications; app_idx++) {
            application = &user_list->applications[app_idx];
            offset_base = 0;

            /* Calculate start offset within contexts[] array for this application */
            for (prev_app_idx = 0; prev_app_idx < app_idx; prev_app_idx++) {
                num_contexts = user_list->applications[prev_app_idx].num_contexts;
                /* Prevent integer overflow when accumulating offset (check before addition) */
                if (num_contexts > INT_MAX - offset_base) {
                    dlt_vlog(LOG_ERR, "%s: Integer overflow at app %d (offset: %d, adding: %d)\n",
                             __func__, app_idx, offset_base, num_contexts);
                    return -1;
                }
                offset_base += num_contexts;
            }

            /* Validate offset is within total contexts array bounds */
            if (offset_base < 0 || offset_base > user_list->num_contexts) {
                dlt_vlog(LOG_ERR, "%s: Invalid offset %d for app %d (max: %d)\n",
                         __func__, offset_base, app_idx, user_list->num_contexts);
                continue;
            }

            /* Iterate through all contexts in this application */
            for (ctx_idx = 0; ctx_idx < application->num_contexts; ctx_idx++) {
                context_index = offset_base + ctx_idx;

                /* Bounds check context index before array access */
                if (context_index < 0 || context_index >= user_list->num_contexts) {
                    dlt_vlog(LOG_ERR, "%s: Context index %d out of bounds (max: %d)\n",
                             __func__, context_index, user_list->num_contexts - 1);
                    break;
                }

                context = &(user_list->contexts[context_index]);

                /* Reset all log level statistics counters for this context */
                if (context != NULL) {
                    for (log_idx = (int)DLT_LOG_STATISTICS_START; log_idx <= (int)DLT_LOG_STATISTICS_END; log_idx++) {
                        context->log_level_statistics[log_idx].stat_cnt = 0;
                        context->log_level_statistics[log_idx].stat_size = 0;
                    }
                }
            }
        }
    }

    return 0;
}
#endif
